'use client';

import React, { useState, useRef, useEffect } from 'react';

type RemoveResponse = {
  ok: boolean;
  usedEndpoint?: string;
  cutoutUrl?: string;
  error?: string;
};

export default function Page() {
  const [file, setFile] = useState<File | null>(null);
  const [bgMode, setBgMode] = useState<'transparent'|'color'|'image'>('transparent');
  const [bgColor, setBgColor] = useState('#ffffff');
  const [bgImage, setBgImage] = useState<File | null>(null);
  const [status, setStatus] = useState<string>('');
  const [usedEndpoint, setUsedEndpoint] = useState<string>('');
  const [cutoutUrl, setCutoutUrl] = useState<string>('');
  const [finalDataUrl, setFinalDataUrl] = useState<string>('');

  const canvasRef = useRef<HTMLCanvasElement>(null);
  const cutoutImgRef = useRef<HTMLImageElement>(null);
  const bgImgRef = useRef<HTMLImageElement>(null);

  useEffect(() => {
    if (!cutoutUrl) return;
    compose();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [cutoutUrl, bgMode, bgColor, bgImage]);

  const onSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!file) { alert('Pilih gambar dulu'); return; }
    setStatus('Mengunggah & memproses...');
    setFinalDataUrl('');
    setCutoutUrl('');
    setUsedEndpoint('');

    const fd = new FormData();
    fd.append('image', file);

    const res = await fetch('/api/removebg', { method: 'POST', body: fd, cache: 'no-store' });
    const data: RemoveResponse = await res.json();
    if (!data.ok) {
      setStatus('Gagal: ' + (data.error || 'unknown'));
      return;
    }
    setUsedEndpoint(data.usedEndpoint || '');
    setCutoutUrl(data.cutoutUrl || '');
    setStatus('Selesai. Pilih background & unduh.');
  };

  const compose = async () => {
    if (!cutoutUrl || !canvasRef.current) return;
    const canvas = canvasRef.current;
    const ctx = canvas.getContext('2d');
    if (!ctx) return;

    // load cutout
    const cut = await loadImage(cutoutUrl);
    canvas.width = cut.width; canvas.height = cut.height;

    if (bgMode === 'transparent') {
      ctx.clearRect(0, 0, canvas.width, canvas.height);
      ctx.drawImage(cut, 0, 0);
    } else if (bgMode === 'color') {
      ctx.fillStyle = bgColor || '#ffffff';
      ctx.fillRect(0, 0, canvas.width, canvas.height);
      ctx.drawImage(cut, 0, 0);
    } else if (bgMode === 'image') {
      if (!bgImage) {
        // no bg image yet; just draw cutout on transparent
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        ctx.drawImage(cut, 0, 0);
      } else {
        const bgURL = URL.createObjectURL(bgImage);
        const bg = await loadImage(bgURL);
        // cover fit
        const { sx, sy, sw, sh, dx, dy, dw, dh } = cover(bg.width, bg.height, canvas.width, canvas.height);
        ctx.drawImage(bg, sx, sy, sw, sh, dx, dy, dw, dh);
        ctx.drawImage(cut, 0, 0);
        URL.revokeObjectURL(bgURL);
      }
    }

    setFinalDataUrl(canvas.toDataURL('image/png'));
  };

  function cover(srcW: number, srcH: number, dstW: number, dstH: number) {
    const scale = Math.max(dstW/srcW, dstH/srcH);
    const sw = Math.floor(dstW/scale);
    const sh = Math.floor(dstH/scale);
    const sx = Math.floor((srcW - sw)/2);
    const sy = Math.floor((srcH - sh)/2);
    return { sx, sy, sw, sh, dx: 0, dy: 0, dw: dstW, dh: dstH };
  }

  function loadImage(src: string) {
    return new Promise<HTMLImageElement>((resolve, reject) => {
      const img = new Image();
      img.crossOrigin = 'anonymous';
      img.onload = () => resolve(img);
      img.onerror = reject;
      img.src = src;
    });
  }

  return (
    <div className="card">
      <h1>Remove Background — Failover + Editor</h1>
      <p className="small">Endpoint otomatis: pxpic/removebg → v1 → v2 → v3 → v4</p>

      <form onSubmit={onSubmit}>
        <label>Gambar sumber (jpg/png/webp)</label>
        <input type="file" accept="image/*" onChange={e => setFile(e.target.files?.[0] || null)} required />

        <div className="grid" style={{marginTop: 12}}>
          <div className="card">
            <label>Jenis Background</label>
            <select value={bgMode} onChange={e => setBgMode(e.target.value as any)}>
              <option value="transparent">Transparan (PNG)</option>
              <option value="color">Warna solid</option>
              <option value="image">Gambar lain</option>
            </select>
            <label>Warna (hex)</label>
            <input type="text" value={bgColor} onChange={e => setBgColor(e.target.value)} placeholder="#111827" />
            <p className="small">Jika memilih "Gambar lain", unggah gambar latar di bawah.</p>
          </div>
          <div className="card">
            <label>Gambar Latar (opsional)</label>
            <input type="file" accept="image/*" onChange={e => setBgImage(e.target.files?.[0] || null)} />
            <p className="small">Akan diposisikan secara "cover".</p>
          </div>
        </div>

        <div className="toolbar">
          <button className="btn" type="submit">Proses Remove BG</button>
          {cutoutUrl ? <a className="btn secondary" href={cutoutUrl} download>Download Cutout</a> : null}
          {finalDataUrl ? <a className="btn" href={finalDataUrl} download="final.png">Download Final</a> : null}
        </div>
      </form>

      {status && <p className="small" style={{marginTop:8}}>{status} {usedEndpoint && <span className="badge">Endpoint: {usedEndpoint}</span>}</p>}

      <div className="grid" style={{marginTop: 16}}>
        <div className="card">
          <h3>Cutout</h3>
          {cutoutUrl ? <img className="img" src={cutoutUrl} alt="cutout" /> : <p className="small">Belum ada hasil</p>}
        </div>
        <div className="card">
          <h3>Preview Final</h3>
          <canvas ref={canvasRef} className="img" />
        </div>
      </div>
    </div>
  );
}
