import { NextRequest, NextResponse } from 'next/server';
import fs from 'node:fs/promises';
import path from 'node:path';
import { Buffer } from 'node:buffer';

export const runtime = 'nodejs';

const UPLOAD_DIR = path.join(process.cwd(), 'tmp', 'uploads');
const CUTOUT_DIR = path.join(process.cwd(), 'tmp', 'cutouts');

const ENDPOINTS = [
  'https://api.nekolabs.my.id/tools/pxpic/removebg',
  'https://api.nekolabs.my.id/tools/remove-bg/v1',
  'https://api.nekolabs.my.id/tools/remove-bg/v2',
  'https://api.nekolabs.my.id/tools/remove-bg/v3',
  'https://api.nekolabs.my.id/tools/remove-bg/v4',
];

function pngSignature(buf: Buffer) {
  const sig = Buffer.from([0x89,0x50,0x4E,0x47,0x0D,0x0A,0x1A,0x0A]);
  return buf.slice(0,8).equals(sig);
}

async function ensureDirs() {
  await fs.mkdir(UPLOAD_DIR, { recursive: true });
  await fs.mkdir(CUTOUT_DIR, { recursive: true });
}

export async function POST(req: NextRequest) {
  try {
    await ensureDirs();
    const form = await req.formData();
    const file = form.get('image');
    if (!file || !(file instanceof File)) {
      return NextResponse.json({ ok:false, error: 'No file' }, { status: 400 });
    }
    const buf = Buffer.from(await file.arrayBuffer());
    const id = crypto.randomUUID().replace(/-/g,'');
    const ext = (file.type.includes('png') ? 'png' : file.type.includes('webp') ? 'webp' : 'jpg');
    const fileName = `${id}.${ext}`;
    const uploadPath = path.join(UPLOAD_DIR, fileName);
    await fs.writeFile(uploadPath, buf);

    // Build public URL for upstream: /api/uploads/[id]
    const urlBase = new URL(req.url);
    const publicImageUrl = `${urlBase.origin}/api/uploads/${fileName}`;

    // Try endpoints in order
    let usedEndpoint = '';
    let cutoutBuf: Buffer | null = null;

    for (const ep of ENDPOINTS) {
      const url = `${ep}?image=${encodeURIComponent(publicImageUrl)}`;
      const res = await fetch(url, {
        method: 'GET',
        headers: { 'Accept': 'image/png,image/*;q=0.9,application/json;q=0.8' },
        cache: 'no-store',
      }).catch(() => null);
      if (!res) continue;
      if (!res.ok) continue;

      const ct = res.headers.get('content-type') || '';
      if (ct.includes('image/')) {
        const ab = await res.arrayBuffer();
        const b = Buffer.from(ab);
        if (pngSignature(b)) { cutoutBuf = b; usedEndpoint = ep; break; }
      } else if (ct.includes('application/json')) {
        const json = await res.json().catch(() => null) as any;
        if (json) {
          const candidate = json.url || json.result || json.data || json.image || json.output;
          if (typeof candidate === 'string') {
            const r2 = await fetch(candidate, { cache: 'no-store' }).catch(() => null);
            if (r2 && r2.ok) {
              const ab2 = await r2.arrayBuffer();
              const b2 = Buffer.from(ab2);
              if (pngSignature(b2)) { cutoutBuf = b2; usedEndpoint = ep + ' (via JSON URL)'; break; }
            }
          }
        }
      }
    }

    if (!cutoutBuf) {
      return NextResponse.json({ ok:false, error: 'All endpoints failed' }, { status: 502 });
    }

    const cutName = `${id}-cutout.png`;
    const cutPath = path.join(CUTOUT_DIR, cutName);
    await fs.writeFile(cutPath, cutoutBuf);

    // Return a local API URL to stream the cutout
    const cutoutUrl = `${new URL(req.url).origin}/api/files/cutouts/${cutName}`;
    return NextResponse.json({ ok:true, usedEndpoint, cutoutUrl });
  } catch (e: any) {
    return NextResponse.json({ ok:false, error: e?.message || 'Server error' }, { status: 500 });
  }
}
